package easik.sketch;

import java.awt.geom.Rectangle2D;
import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Set;

import javax.swing.tree.DefaultMutableTreeNode;

import org.jgraph.JGraph;
import org.jgraph.graph.AttributeMap;
import org.jgraph.graph.DefaultGraphCell;
import org.jgraph.graph.GraphConstants;
import org.jgrapht.graph.DefaultEdge;
import org.jgrapht.graph.ListenableDirectedGraph;

import easik.Easik;
import easik.sketch.constraint.Constraint;
import easik.sketch.datatype.DataType;
import easik.sketch.datatype.DataTypeController;
import easik.sketch.document.DocumentInfo;
import easik.sketch.edge.GuideEdge;
import easik.sketch.edge.SketchEdge;
import easik.sketch.path.SketchPath;
import easik.sketch.util.SketchAdapter;
import easik.sketch.util.SketchFileIO;
import easik.sketch.util.SketchSelectionListener;
import easik.sketch.util.graph.EasikCellViewFactory;
import easik.sketch.vertex.EntityNode;
import easik.sketch.vertex.SketchVertex;
import easik.states.LoadingState;
import easik.ui.ApplicationFrame;
import easik.ui.SketchUI;
import easik.ui.tree.InfoTreeUI;

/** 
 * A Sketch represents an EA Sketch diagram used to represent a database, this object
 * also extends the JGraph swing component, allowing it to be added directly to the
 * application's GUI.
 * 
 * When done with the current sketch, instead of creating a new one, the sketch should
 * simply be reinitialised, and it will become ready. Since a sketch is also a Swing component
 * it can get hairy creating a new one and changing all the references. This might have
 * become easier since the singletons were introduced, so feel free to try to change that.
 *  
 * @author Rob Fletcher 2005
 * @author Kevin Green 2006
 * @author Vera Ranieri 2006
 * @version 2006-08-02 Kevin Green
 */
public class Sketch extends JGraph {

	/**
	 * Records whether the sketch has been modified since the last save.  Initialized to <b>false</b>
	 */
	private boolean _dirty = false;
	/**
	 * The current file, initialized to <b>null</b>
	 */
	private File _currentFile = null;
	/**
	 * The current ListenableDirectedGraph
	 */
	private ListenableDirectedGraph<SketchVertex, DefaultEdge> _ourGraph;
	/**
	 * The current SketchAdapter
	 */
	private SketchAdapter _modelAdapter;
	/**
	 * Hash Map of all entity nodes, indexed by their name
	 */
	private HashMap<String, EntityNode> _entityNodes;
	/**
	 * Hash Map of all edges, indexed by their label
	 */
	private HashMap<String, DefaultEdge> _edges;
	/**
	 * Hash Map of all paths, indexed by their name
	 */
	private HashMap<String, SketchPath> _paths;
	/**
	 * Linked List of all constraints
	 */
	private LinkedList<Constraint> _constraints;
	/**
	 * The current DataTypeController for this sketch
	 */
	private DataTypeController _typeController;
	/**
	 * The current SketchSelectionListener
	 */
	private SketchSelectionListener _selectionListener;
	/**
	 * The current ApplicationFrame
	 */
	private ApplicationFrame _appFrame;
	/**
	 * The current DocumentInfo
	 */
	private DocumentInfo _docInfo;

	/**
	 * The default constructor sets all the visual settings for the JGraph, as well as
	 * initialising the sketch to be empty. It also adds appropriate listeners for all 
	 * of the actions we are concerned with.
	 * 
	 * @param inFrame The application frame of the sketch
	 */
	public Sketch(ApplicationFrame inFrame) {
		super();
		_appFrame = inFrame;
		this.setAntiAliased(true);
		this.setDisconnectable(false);
		this.setConnectable(false);
		this.setEditable(false);
		this.setSizeable(false);
		this.getGraphLayoutCache().setFactory(new EasikCellViewFactory());
		this.initialiseSketch();
		_selectionListener = new SketchSelectionListener(this);
		this.getGraphLayoutCache().setAutoSizeOnValueChange(true);
		this.addGraphSelectionListener(_selectionListener);
		
		updateUI();
	}

	/**
	 * Get the working file for this sketch.
	 * @return The file last saved using this sketch
	 */
	public File getFile() {
		return _currentFile;
	}

	/**
	 * This assigns a file to the current sketch.
	 * 
	 * @param inFile File to be assigned.
	 */
	public void setFile(File inFile) {
		_currentFile = inFile;

		if (inFile == null) {
			Easik.getInstance().getFrame().setTitle("Easik - Untitled");
		} else {
			Easik.getInstance().getFrame().setTitle(
				"Easik - " + inFile.getName());
		}
	}

	/**
	 * Since this is a Swing component, this method is overloading a method of
	 * JGraph to adjust the look and feel. The feel we are changing is ignoring all but
	 * left clicks, allowing for right-click functionality not affecting the selections.
	 */
	public void updateUI() {
		this.setUI(new SketchUI());
		this.invalidate();
	}

	/**
	 * Whenever a cell moves, it calls this method of the sketch in order to keep
	 * all of the coordinates stored in the entities accurate. Called from the sketch
	 * listener.
	 * 
	 * @param inCell The cell which has been moved
	 */
	public void updateCellMoved(DefaultGraphCell inCell) {
		_modelAdapter.updateVertexPos(inCell);
	}

	/**
	 * Determines whether the sketch has been modified since the last save.  
	 * 
	 * @return The dirtiness, true means dirty.
	 */
	public boolean getDirty() {
		return _dirty;
	}

	/**
	 * Used to mark a sketch as dirty or not.
	 * 
	 * @param inDirty NEw dirtiness.
	 */
	public void setDirty(boolean inDirty) {
		_dirty = inDirty;
	}
	
	/**
	 * Returns the DataTypeController for the sketch
	 * 
	 * @return The DataTypeController for the sketch
	 */
	public DataTypeController getDataTypeController(){
		return _typeController;
	}

	/**
	 *	Returns the parental application frame to whoever asks for it.
	 * @return The current application frame
	 */
	public ApplicationFrame getFrame() {
		return _appFrame;
	}
	
	/**
	 * Gets the document information
	 * @return The document information 
	 */
	public DocumentInfo getDocInfo(){
		return _docInfo;
	}

	/**
	 * When we initialise the sketch, we flush out all the data concerning the sketch
	 * itself. Even the modelAdapter is reinitialized.
	 *
	 *	This methods serves as a "new sketch" function. 
	 */
	public void initialiseSketch() {
		_entityNodes = new HashMap<String, EntityNode>();
		_edges = new HashMap<String, DefaultEdge>();
		_paths= new HashMap<String, SketchPath>();
		_constraints = new LinkedList<Constraint>();
		_ourGraph = new ListenableDirectedMultigraph<SketchVertex, DefaultEdge>(DefaultEdge.class);
		_modelAdapter = new SketchAdapter(_ourGraph, this);
		_modelAdapter.addGraphModelListener(new SetPoints());
		this.setModel(_modelAdapter);
		if(_appFrame.getInfoTreeUI() != null)
		{
			_appFrame.setInfoTreeUI(new InfoTreeUI()); //Wipe Tree
			_appFrame.getInfoTreeUI().refreshTree();
		}
		_typeController = new DataTypeController();
		_docInfo = new DocumentInfo();
	}

	/**
	 * When we initialise a new sketch, we need to clear the selection
	 * buffer just in case something is still selected. Or else it will remain
	 * selected because there will be no events removing it.	 
	 */
	public void newSketch() {
		initialiseSketch();
		this._selectionListener.emptySelection();
		this.setFile(null);
		_appFrame.refreshPlatformCheckBoxes();
	}

	/**
	 * Used to initialise a new sketch based on provided data (usually from the Sketch loading
	 * methods).
	 * 
	 * @param entityNodes A hash map of all of the entities in the sketch
	 * @param edges A hash map containing all of the edges in the sketch
	 * @param paths A hash map containing all of the paths in the sketch
	 * @param constraints A linked list containing all of the constraints of the sketch
	 * @param dataTypes An array list of all the data types used for the sketch
	 * @param inMySQL If the MySQL output is enabled or not
	 * @param inOracle If Oracle output is enabled or not
	 * @param inDB2 If DB2 output is enabled or not
	 * @param inXML If XML output is enabled or not
	 * @param inUserDefined If user defined output is enabled or not
	 * @param head The header created from the loaded XML file.
	 */
	public void initialiseFromData(
		HashMap<String, EntityNode> entityNodes,
		HashMap<String, DefaultEdge> edges,
		HashMap<String, SketchPath> paths,
		LinkedList<Constraint> constraints,
		ArrayList dataTypes,
		DocumentInfo head,
		boolean inMySQL,
		boolean inOracle,
		boolean inDB2,
		boolean inXML,
		boolean inUserDefined) 
	{
		initialiseSketch();
		
		_entityNodes = entityNodes;
		_edges = edges;
		_paths = paths;
		_constraints = constraints;
		_docInfo = head;
		
		_appFrame.setTreeName(_docInfo.getName());
		
		//Set data type platforms
		_typeController.set_useMySQL(inMySQL);
		_typeController.set_useOracle(inOracle);
		_typeController.set_useDB2(inDB2);
		_typeController.set_useXML(inXML);
		_typeController.set_useUserDefined(inUserDefined);
		
		_appFrame.refreshPlatformCheckBoxes();
		
		//Add datatypes
		_typeController.getDataTypes().clear();
		int numTypes = dataTypes.size();
		if(numTypes>0){
			for(int i=0; i<numTypes; i++){
				_typeController.addDataType((DataType)dataTypes.get(i));
			}
			_typeController.loadNullsFromDefaults("ONLY NULL");
		}
		else{
			_typeController.resetDataTypes();
		}
		
		//Add the entities
		Iterator iterator = _entityNodes.keySet().iterator();
		while (iterator.hasNext()) {
			EntityNode currentEntity = (EntityNode) (_entityNodes.get(iterator.next()));
			if (currentEntity != null) {
				this.addEntity(currentEntity);
			}
		}

		//Add the edges
		iterator = _edges.keySet().iterator();
		while (iterator.hasNext()) {
			SketchEdge currentEdge = (SketchEdge) (_edges.get(iterator.next()));
			_ourGraph.addEdge((SketchVertex)currentEdge.getSourceObj(), (SketchVertex)currentEdge.getTargetObj(), currentEdge);
		}

		//Add the constraints
		for (Constraint c :_constraints) {
			this.addConstraint(c);
		}
		Easik.getInstance().getStateManager().peekState().selectionUpdated();
	}

	/**
	 * An accessor for the model's JGraphT->JGraph adapter.
	 * 
	 * @return The current model adapter
	 */
	public SketchAdapter getAdapter() {
		return _modelAdapter;
	}

	/**
	 * An accessor for the Model's selection listener
	 * @return The selection listener
	 */
	public SketchSelectionListener getSelectionListener() {
		return _selectionListener;
	}

	/**
	 * Get the sketch's underlying JGraphT graph, a ListenableDirectedGraph
	 * 
	 * @return The underlying graph
	 */
	public ListenableDirectedGraph<SketchVertex, DefaultEdge> getGraphData() {
		return _ourGraph;
	}

	/**
	 * Returns the HashMap of all of the edges in the sketch
	 * 
	 * @return HashMap of all the edges.
	 */
	public HashMap getEdges() {
		return _edges;
	}
	
	/**
	 * Returns the HashMap of all paths in the sketch
	 * 
	 * @return HashMap of paths
	 * @since 2006-05-29
	 */
	public HashMap getPaths(){
		return _paths;
	}

	/**
	 * Accessor for the entities in the sketch
	 * 
	 * @return HashMap of the entities
	 */
	public HashMap getEntities() {
		return _entityNodes;
	}

	/**
	 * Returns the linkedlist of constraints
	 * 
	 * @return The constraints
	 */
	public LinkedList<Constraint> getConstraints() {
		return _constraints;
	}

	// TODO: For loading and saving, have it return its success or an error message
	/**
	 * Requests that an XML file be loaded into the sketch.	
	 * 
	 * @param inputFile The file from which the data will be drawn.
	 */
	public void loadFromXML(File inputFile) {
		SketchFileIO.graphicalSketchFromXML(inputFile, this);
	}

	/**
	 * Removes an entity, and also cascades to remove all the arrows 
	 * involved with it.
	 * 
	 * @param toRemove The entity about to be removed
	 */
	public void removeEntity(EntityNode toRemove) {

		// Check for edges that use these entities
		Set<String> keys = _edges.keySet();
		
		//So we don't get a ConcurrentModificationException
		ArrayList<SketchEdge> removeEdges = new ArrayList<SketchEdge>();
		
		for(String current: keys) {
			SketchEdge currentEdge = (SketchEdge)_edges.get(current);
			if ((Easik.getInstance().getFrame().getSketch().getGraphData().getEdgeSource(currentEdge).equals(toRemove))
				|| (Easik.getInstance().getFrame().getSketch().getGraphData().getEdgeTarget(currentEdge).equals(toRemove))) {
				//add the edge to a list of edges to remove
				removeEdges.add(currentEdge);
			}
		}

		//Remove the edges
		for(SketchEdge e :removeEdges){
			removeEdge(e);
		}
		_entityNodes.remove(toRemove.toString());
		_ourGraph.removeVertex(toRemove);
		
		//Remove Entity from tree
		_appFrame.getInfoTreeUI().removeEntity(toRemove);
	}

	/**
	 * Removes an edge, also cascades to remove all constraints using it.
	 * 
	 * @param toRemove The edge about to be removed
	 */
	public void removeEdge(SketchEdge toRemove) {

		// Check for constraints that need these edges
		for (int i = 0; i < _constraints.size(); i++) {
			Constraint currentConstraint = (Constraint) _constraints.get(i);
			if (currentConstraint.hasEdge(toRemove)) {
				removeConstraint(currentConstraint);
				i--;
			}
		}

		_edges.remove(toRemove.getName());
		_ourGraph.removeEdge(toRemove);
	}

	/**
	 * Removes a constraint and guide arrows
	 * 
	 * @param toRemove The constraint about to be removed
	 */
	public void removeConstraint(Constraint toRemove) {
		ArrayList<SketchEdge> edges = toRemove.getEdges();
		for (DefaultEdge e: edges) {
			GuideEdge toClear =
				(GuideEdge) _ourGraph.getEdge(
					toRemove,
					(SketchVertex)Easik.getInstance().getFrame().getSketch().getGraphData().getEdgeSource(e));
			_ourGraph.removeEdge(toClear);
		}
		_ourGraph.removeVertex(toRemove);
		_constraints.remove(toRemove);
		
		DefaultMutableTreeNode parent = (DefaultMutableTreeNode) toRemove.getNode().getParent();
		toRemove.getNode().removeFromParent();
		
		_appFrame.getInfoTreeUI().refreshTree(parent);
	}

	/**
	 * Saves the existing sketch as an XML file.
	 * 	
	 * @param outputFile The file to be written to	   
	 */
	public void saveToXML(File outputFile) {
		SketchFileIO.sketchToXML(outputFile, this);
	}

	/**
	 * Add a new, empty entity at point X, Y 
	 *
	 * @param name The name of the new entity being added
	 * @param x X Coordinate of new entity
	 * @param y Y Coordinate of new entity
	 */
	public void addNewEntity(String name, double x, double y) {
		EntityNode newEntity;
		newEntity = new EntityNode(name, (int) x, (int) y);
		addEntity(newEntity);
	}

	/**
	 * Add an entity to the graph, deals with all of the dependencies.
	 * 
	 * @param theEntity
	 */
	public void addEntity(EntityNode theEntity) {
		//Push loading state
		Easik.getInstance().getStateManager().pushState(new LoadingState());
		
		// Add our entity to the graph				
		_ourGraph.addVertex(theEntity);
		// Add our entity to our table of entities
		_entityNodes.put(theEntity.toString(), theEntity);
		// Set the on-screen position of our entity to the attributes of the entity
		Rectangle2D bounds =
			GraphConstants.getBounds(
				getAdapter().getVertexCell(theEntity).getAttributes());
		bounds.setRect(
			theEntity.getX(),
			theEntity.getY(),
			bounds.getHeight(),
			bounds.getHeight());
		// Reload the graph to respect the new changes
		getGraphLayoutCache().reload();
		
		//Add Entity to tree
		_appFrame.getInfoTreeUI().addEntity(theEntity);
		
		//Pop state
		Easik.getInstance().getStateManager().popState();
	}

	/**
	 * Adds an edge to the sketch
	 * 
	 * @param inEdge The edge to be added
	 */
	public void addEdge(SketchEdge inEdge) {
		//Push loading state
		Easik.getInstance().getStateManager().pushState(new LoadingState());
		
		_edges.put(inEdge.getName(), inEdge);
		_ourGraph.addEdge((SketchVertex)inEdge.getSourceObj(), (SketchVertex)inEdge.getTargetObj(), inEdge);
		
		//Pop state
		Easik.getInstance().getStateManager().popState();
	}

	/**
	 * Add a new constraint to the sketch
	 * 
	 * @param c The constraint to be added
	 */
	public void addNewConstraint(Constraint c){
		
		ArrayList<SketchPath> paths = c.getPaths();
		for(SketchPath p : paths){
			_paths.put(p.getId(), p);
		}
		_constraints.addLast(c);
		addConstraint(c);
	}
	/**
	 * Adds a constraint to the sketch. This will register the constraint in the 
	 * constraint list, as well as adding a visual representation of the constraint
	 * to the graph.
	 * 
	 * @param c The constraint to be added.
	 */
	public void addConstraint(Constraint c) {
		//Push loading state
		Easik.getInstance().getStateManager().pushState(new LoadingState());
		
		c.setVisible(c.isVisible());
		getGraphLayoutCache().reload();
		_appFrame.getInfoTreeUI().addConstraint(c);

		//Pop state
		Easik.getInstance().getStateManager().popState();
	}

	/**
	* Returns the next available 'NewName', so we don't get duplicates.
	* 
	* @return The next new name.
	*/
	public String getNewName() {
		int num = 0;
		while (isNameUsed("NewEntity" + num)) {
			num++;
		}
		return "NewEntity" + num;
	}

	/**
	* Checks to see if a name is in use, so that we will not have several instances
	* at once.
	* 
	* @param inName The desired new name to check against
	* @return Is it used or not.
	*/
	public boolean isNameUsed(String inName) {
		
		Set<String> keys = _entityNodes.keySet();
		for(String name : keys){
			if(name.equalsIgnoreCase(inName))
				return true;
		}
		
		return false;
	}

	/**
	* Returns the first available unique edge identifier
	* @param isInjective true if the edge is injective, false otherwise
	* @return The next id
	*/
	public String getNewEdgeName(boolean isInjective) {
		int currentID = 0;
		boolean foundOne = false;
		
		String prefix;
		if(isInjective)
			prefix = "isA_";
		else
			prefix = "f";

		while (!foundOne) {
			currentID++;
			if (!isEdgeNameUsed(prefix + currentID)) {
				foundOne = true;
			}
		}
		return prefix + currentID;
	}

	/**
	 * Checks to see if a name is in use, so that we will not have several instances
	 * at once. For edges.
	 * 
	 * @param inName The desired new edge name to check against
	 * @return Is it used or not.
	 */
	public boolean isEdgeNameUsed(String inName) {
		
		Set<String> edges = _edges.keySet();
		for(String e : edges){
			if(e.equalsIgnoreCase(inName))
				return true;
		}
		
		return false;
	}
	
	/**
	 * Tests if the sketch has multiple edges between any two entities
	 * 
	 * @return True if there exists a situation of multiple edges
	 */
	public boolean containsMultiEdges(){
		//Create array to store edges in based on source and target
		ArrayList entities = new ArrayList(Easik.getInstance().getFrame().getSketch().getEntities().values());
		int numVer = entities.size(); 
		ArrayList[][] edgeArr = new ArrayList[numVer][numVer];
		for(int i=0; i<numVer; i++){
			for(int j=i; j<numVer; j++){
				ArrayList temp = new ArrayList();
				edgeArr[i][j] = temp;
				edgeArr[j][i] = temp;
			}
		}
		
		//Add edges to array
		Object entireSet[] = Easik.getInstance().getFrame().getSketch().getRoots();
		for(int i=0; i<entireSet.length; i++){
			DefaultGraphCell curObject = (DefaultGraphCell) entireSet[i];
			AttributeMap myMap = ((DefaultGraphCell)entireSet[i]).getAttributes();
			if(curObject.getUserObject() instanceof SketchEdge){
				SketchEdge curEdge = (SketchEdge) curObject.getUserObject();
				edgeArr[entities.indexOf(curEdge.getSourceObj())][entities.indexOf(curEdge.getTargetObj())].add(curObject);
			}
		}
		
		//Process edges
		for(int i=0; i<numVer; i++){
			for(int j=i; j<numVer; j++){
				if(edgeArr[i][j].size() > 1)
					return true;
			}
		}	
		return false;
	}
}
